<?php

namespace App\Repositories;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

abstract class BaseRepository
{
    protected $model;
    protected $cacheTime = 3600; // 1 hour

    public function __construct(Model $model)
    {
        $this->model = $model;
    }

    public function find($id)
    {
        return Cache::remember($this->getCacheKey('find', $id), $this->cacheTime, function () use ($id) {
            return $this->model->find($id);
        });
    }

    public function findWhere(array $criteria)
    {
        return Cache::remember($this->getCacheKey('findWhere', json_encode($criteria)), $this->cacheTime, function () use ($criteria) {
            return $this->model->where($criteria)->first();
        });
    }

    public function getWhere(array $criteria): Collection
    {
        return Cache::remember($this->getCacheKey('getWhere', json_encode($criteria)), $this->cacheTime, function () use ($criteria) {
            return $this->model->where($criteria)->get();
        });
    }

    public function create(array $data)
    {
        $model = $this->model->create($data);
        $this->clearCache();
        return $model;
    }

    public function update($id, array $data)
    {
        $model = $this->model->find($id);
        if ($model) {
            $model->update($data);
            $this->clearCache();
        }
        return $model;
    }

    public function delete($id)
    {
        $result = $this->model->destroy($id);
        $this->clearCache();
        return $result;
    }

    protected function getCacheKey(string $method, $params): string
    {
        return sprintf(
            '%s_%s_%s_%s',
            strtolower(class_basename($this->model)),
            $method,
            json_encode($params),
            app()->getLocale()
        );
    }

    protected function clearCache(): void
    {
        Cache::tags([strtolower(class_basename($this->model))])->flush();
    }
}
